package com.plutext.samples.mergepptx;
/*
 *  Copyright 2014, Plutext Pty Ltd.
 *   
 *  This file is part of the MergePptx commercial extension to docx4j.
 *
 *  MergePptx is licensed under the Plutext Component Agreement (the "License"); 
 *  you may not use this file except in compliance with the License. 
 *  
 *  In particular, this source code is CONFIDENTIAL, and you must ensure it
 *  stays that way. 
 *
 *  You may obtain a copy of the License at 
 *
 *      http://www.plutext.com/license/Plutext_MergeDocx_Component_License.pdf 
 *
 *  Unless required by applicable law or agreed to in writing, software 
 *  distributed under the License is distributed on an "AS IS" BASIS, 
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 *  See the License for the specific language governing permissions and 
 *  limitations under the License.
 */


import java.io.File;

import net.engio.mbassy.bus.MBassador;
import net.engio.mbassy.bus.config.BusConfiguration;
import net.engio.mbassy.bus.config.Feature;
import net.engio.mbassy.bus.error.IPublicationErrorHandler;

import org.docx4j.events.Docx4jEvent;
import org.docx4j.openpackaging.packages.OpcPackage;
import org.docx4j.openpackaging.packages.PresentationMLPackage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.plutext.merge.pptx.PresentationBuilder;
import com.plutext.merge.pptx.SlideRange;
import com.plutext.merge.pptx.ThemeTreatment;

/**
 * Merge/concatenate/append/join pptx files.
 * 
 * @author jharrop
 *
 */
public class MergeWholePresentations {
		
	public final static String DIR_IN = System.getProperty("user.dir")+ "/src/main/resources/sample-pptx/";
	public final static String DIR_OUT = System.getProperty("user.dir")+ "/";
	
	/*
	 * With value IGNORE, the first layout with a given name 
	 * (for example, "Title Slide"), will be used for all slides
	 * (from all input presentations) using that layout name.
	 * 
	 * With value RESPECT, the Theme name will be used as a 
	 * namespace for layout names.  For this to work,
	 * you need a different Theme name in each deck;
	 * if that isn't the case, try RENAME.
	 * 
	 * In Powerpoint, you can see your theme and layout names
	 * by clicking on View > Slide Master.
	 * 
	 * RENAME is lke RESPECT, except the Theme names in each deck are made unique.	 */
	public final static ThemeTreatment themeTreatment = ThemeTreatment.RENAME;
	
	protected static Logger log = LoggerFactory.getLogger(MergeWholePresentations.class);
		
	private final static boolean listen = false;
	
	public static void main(String[] args) throws Exception {
		
		long startMS = System.currentTimeMillis();
		
		String[] deck = {
				"javase.pptx" ,"microsoft1.pptx", "oracle.pptx", "SmartArt.pptx"
		};				

		ListeningBean listener;
		if (listen) {
			// Creation of message bus
			MBassador<Docx4jEvent> bus = new MBassador<Docx4jEvent>(
					new BusConfiguration()
				     .addFeature(Feature.SyncPubSub.Default()) // configure the synchronous message publication
				     .addFeature(Feature.AsynchronousHandlerInvocation.Default()) // configure asynchronous invocation of handlers
				     .addFeature(Feature.AsynchronousMessageDispatch.Default()) // configure asyncronous message publication (fire&forget)
				     .addPublicationErrorHandler(new IPublicationErrorHandler.ConsoleLogger())
				     );
			
			Docx4jEvent.setEventNotifier(bus);
			//  and registration of listeners
			listener = new ListeningBean();
			bus.subscribe(listener);					
		}
		
		PresentationBuilder builder = new PresentationBuilder();		
		builder.setThemeTreatment(themeTreatment);
		
		for (int i=0 ; i< deck.length; i++) {
			
			log.info("\n\n loading " + i + "  " + deck[i] + "\n\n");
		
			// Create a SlideRange representing the slides in this pptx
			SlideRange sr = new SlideRange(
					(PresentationMLPackage)OpcPackage.load(
											new File(DIR_IN + deck[i])));
			sr.setName(i+ " " + deck[i]);  // PkgIdentifier for ListeningBean
			
			// Add the slide range to the output
			builder.addSlideRange(sr);
		
		}
		
		
		builder.getResult().save(
				new File(DIR_OUT + "/OUT_MergeWholePresentations_" + themeTreatment +".pptx"));
		
		if (listen) {
			System.out.println(listener.sw);
		}
		
		// The times above don't include various things, such as Context init, and saving the final docx
		long elapsedMS = System.currentTimeMillis() - startMS;
		int secs = Math.round(elapsedMS/1000);
		//System.out.println("time taken: " + secs + "secs");
		
		log.info("Done.  Saved " + "/OUT_MergeWholePresentations_" + themeTreatment +".pptx");
		
	}
	
	
}
	
