package com.plutext.samples.ole;
/*
 *  Copyright 2016, Plutext Pty Ltd.
 *   
 *  This sample file is licensed under the Apache License, Version 2.0 (the "License"); 
    you may not use this file except in compliance with the License. 

    You may obtain a copy of the License at 

        http://www.apache.org/licenses/LICENSE-2.0 

    Unless required by applicable law or agreed to in writing, software 
    distributed under the License is distributed on an "AS IS" BASIS, 
    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
    See the License for the specific language governing permissions and 
    limitations under the License.

 */




import java.io.File;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.docx4j.TraversalUtil.CallbackImpl;
import org.docx4j.XmlUtils;
import org.docx4j.openpackaging.packages.WordprocessingMLPackage;
import org.docx4j.openpackaging.parts.WordprocessingML.MainDocumentPart;
import org.docx4j.wml.Document;
import org.docx4j.wml.P;
import org.docx4j.wml.R;
import org.docx4j.wml.Text;

import com.plutext.ole.EmbeddingType;
import com.plutext.ole.OleHelperDocx;

/**
 * Example of how to link or embed an object at an identified location
 * in a docx.
 * 
 * This is achieved by using the API to return a run, which can be
 * inserted anywhere in the Main Document Part where a run is allowed.
 * 
 * So, its just a matter of finding the place in the document where
 * you want to insert that run.  Various "markers" can be used to specify
 * the spot in the document, eg:
 * 
 * - text in the run (used here for simplicity)
 * - a content control with a specific tag
 * - a bookmark
 * - a custom field
 * 
 * The 2 basic techniques in docx4j for finding content are Traversal,
 * and XPath.  Here we use traversal.
 * 
 * In this example, we embed an xlsx.
 * 
 * Note: OLE insertion of XML based Office formats (eg docx/pptx/xlsx) is a bit
 * different to how other OLE objects are handled.
 * 
 * @author Jason Harrop
 */
public class Docx_ole_Xlsx_IdentifiedLocation {
	
	
	// The string we'll replace.  If you use this technique,
	// make sure your string is not split across runs!
	private static String IDENTIFIER = "[XLS FILE]";
	

	static org.docx4j.wml.ObjectFactory wmlObjectFactory = org.docx4j.jaxb.Context.getWmlObjectFactory();
	
	public static void main(String[] args) throws Exception {
		
				
		////////////////////////////////////////////////////////
		// For this sample, create a docx package using docx4j

		WordprocessingMLPackage wordMLPackage = WordprocessingMLPackage.createPackage();
		MainDocumentPart mdp = wordMLPackage.getMainDocumentPart();
		
		// Add sample content, 
		Document document = (Document)XmlUtils.unmarshalString(openXML);	
		mdp.setJaxbElement(document);

		
		////////////////////////////////////////////////////////
		// OLE config

		EmbeddingType embeddingType = EmbeddingType.XLSX;

		// Link or embed?
		boolean OLE_LINK_ONLY = false;
		
		// Embed
		byte[] contentBytes = FileUtils.readFileToByteArray(
				new File(System.getProperty("user.dir")
						+"/src/main/resources/simple.xlsx" ));
		
        // or link 
        String targetURL = "file:///F:/simple.xlsx";    // the actual location

		// the image on the document surface
		byte[] imageBytes = FileUtils.readFileToByteArray(new File(System.getProperty("user.dir")+"/greentick.png" ));
		
		// if you know your image mime type, you can avoid image introspection (which avoids temp file creation)
		String mime = "image/png"; // .. otherwise, set this to null
		
		String caption = "the caption"; // can be anything, but may be used if displaying content rather than icon 		
		

		
		////////////////////////////////////////////////////////
		// Perform OLE
		
		OleHelperDocx OleHelperDocx = new OleHelperDocx(wordMLPackage);
			
		R oleRun = null;
		
		if (OLE_LINK_ONLY) { 

			oleRun = OleHelperDocx.linkOffice(embeddingType, 
					caption,  
					targetURL, imageBytes);
			
		} else {
			oleRun = OleHelperDocx.embedOffice(embeddingType, 
						caption,  
						contentBytes, imageBytes);
		}
		
		////////////////////////////////////////////////////////
		// Replace the text with oleRun
		InsertionPointLocator insertionPointLocator = new InsertionPointLocator(IDENTIFIER);
		insertionPointLocator.walkJAXBElements(document);
		
		if (insertionPointLocator.foundR==null) {
			System.out.println("FAILED to find insertion point '" + IDENTIFIER + "'; exiting.");
			return;
		}

		// Looks like we found the target run
		List contentList = insertionPointLocator.foundList;
		int index = contentList.indexOf(insertionPointLocator.foundR);
//		int index = getIndex(contentList, insertionPointLocator.foundR);
		if (index <0 ) {
			// shouldn't happen
			System.out.println("FAILED to find index of insertion point; exiting.");
		}
		contentList.remove(index); // assume the run we found contains just the IDENTIFIER and nothing else, so it can safely be replaced in its entirety
		contentList.add(index, oleRun);
		
		// All done
		
		    // Save the resulting docx
			wordMLPackage.save(new File(System.getProperty("user.dir")
					+ "/OUT_xlsx_embedded_at_identifier.docx"));
			
	}
			
	public static class InsertionPointLocator extends CallbackImpl {
		
		private String textToFind; 
		
		public InsertionPointLocator(String textToFind) {
		
			this.textToFind = textToFind;
		}

		private List foundList = null;
		public List getFoundList() {
			return foundList;
		}

		private R foundR = null;
		public R getFoundR() {
			return foundR;
		}

		
		private List currentList = null;
		private R currentR = null;
		
		private boolean done = false;
		
		@Override
		public List<Object> apply(Object wrapped) {
			
			Object o = XmlUtils.unwrap(wrapped);

			if (o instanceof P) {
				
				currentList = ((P)o).getContent();
			
			} else if (o instanceof R) {
				
				currentR = (R)o;
				
			} else if (o instanceof Text) {
				
				Text t = (Text)o;
				if (t.getValue()!=null
						&& t.getValue().equals(textToFind)) {
					// Got it!
					foundList = currentList;
					foundR = currentR;
					done = true;
				}
			}
			return null;
		}
		
		public boolean shouldTraverse(Object o) {
			return !done;
		}		

	}	
	
	
	// Sample content, containing IDENTIFIER
	private static String openXML = "<w:document mc:Ignorable=\"w14 wp14\" xmlns:w=\"http://schemas.openxmlformats.org/wordprocessingml/2006/main\" xmlns:w14=\"http://schemas.microsoft.com/office/word/2010/wordml\" xmlns:mc=\"http://schemas.openxmlformats.org/markup-compatibility/2006\" xmlns:wp14=\"http://schemas.microsoft.com/office/word/2010/wordprocessingDrawing\">"
            + "<w:body>"
                    + "<w:p>"
                            + "<w:pPr>"
                                    + "<w:rPr>"
                                            + "<w:lang w:val=\"en-AU\"/>"

                                    + "</w:rPr>"

                            + "</w:pPr>"

                            + "<w:r>"
                                    + "<w:rPr>"
                                            + "<w:lang w:val=\"en-AU\"/>"

                                    + "</w:rPr>"

                                    + "<w:t>Some content</w:t>"

                            + "</w:r>"

                    + "</w:p>"

                    + "<w:tbl>"
                            + "<w:tblPr>"
                                    + "<w:tblStyle w:val=\"TableGrid\"/>"

                                    + "<w:tblW w:type=\"auto\" w:w=\"0\"/>"

                                    + "<w:tblLook w:firstColumn=\"1\" w:firstRow=\"1\" w:lastColumn=\"0\" w:lastRow=\"0\" w:noHBand=\"0\" w:noVBand=\"1\" w:val=\"04A0\"/>"

                            + "</w:tblPr>"

                            + "<w:tblGrid>"
                                    + "<w:gridCol w:w=\"3192\"/>"

                                    + "<w:gridCol w:w=\"3192\"/>"

                                    + "<w:gridCol w:w=\"3192\"/>"

                            + "</w:tblGrid>"

                            + "<w:tr>"
                                    + "<w:tc>"
                                            + "<w:tcPr>"
                                                    + "<w:tcW w:type=\"dxa\" w:w=\"3192\"/>"

                                            + "</w:tcPr>"

                                            + "<w:p>"
                                                    + "<w:pPr>"
                                                            + "<w:rPr>"
                                    + "<w:lang w:val=\"en-AU\"/>"

                                                            + "</w:rPr>"

                                                    + "</w:pPr>"

                                            + "</w:p>"

                                    + "</w:tc>"

                                    + "<w:tc>"
                                            + "<w:tcPr>"
                                                    + "<w:tcW w:type=\"dxa\" w:w=\"3192\"/>"

                                            + "</w:tcPr>"

                                            + "<w:p>"
                                                    + "<w:pPr>"
                                                            + "<w:rPr>"
                                    + "<w:lang w:val=\"en-AU\"/>"

                                                            + "</w:rPr>"

                                                    + "</w:pPr>"

                                            + "</w:p>"

                                    + "</w:tc>"

                                    + "<w:tc>"
                                            + "<w:tcPr>"
                                                    + "<w:tcW w:type=\"dxa\" w:w=\"3192\"/>"

                                            + "</w:tcPr>"

                                            + "<w:p>"
                                                    + "<w:pPr>"
                                                            + "<w:rPr>"
                                    + "<w:lang w:val=\"en-AU\"/>"

                                                            + "</w:rPr>"

                                                    + "</w:pPr>"

                                            + "</w:p>"

                                    + "</w:tc>"

                            + "</w:tr>"

                            + "<w:tr>"
                                    + "<w:tc>"
                                            + "<w:tcPr>"
                                                    + "<w:tcW w:type=\"dxa\" w:w=\"3192\"/>"

                                            + "</w:tcPr>"

                                            + "<w:p>"
                                                    + "<w:pPr>"
                                                            + "<w:rPr>"
                                    + "<w:lang w:val=\"en-AU\"/>"

                                                            + "</w:rPr>"

                                                    + "</w:pPr>"

                                            + "</w:p>"

                                    + "</w:tc>"

                                    + "<w:tc>"
                                            + "<w:tcPr>"
                                                    + "<w:tcW w:type=\"dxa\" w:w=\"3192\"/>"

                                            + "</w:tcPr>"

                                            
                                            + "<w:p>"
                                                    + "<w:pPr>"
                                                            + "<w:rPr>"
                                    + "<w:lang w:val=\"en-AU\"/>"

                                                            + "</w:rPr>"

                                                    + "</w:pPr>"

		                                            + "<w:r>"
				                                            + "<w:t>before</w:t>"
				                                    + "</w:r>"
                                                    
                                                    + "<w:r>"
                                                            + "<w:rPr>"
                                    + "<w:lang w:val=\"en-AU\"/>"

                                                            + "</w:rPr>"
                                                            
// Here is where we'll add the OLE object                                                            
                                                            + "<w:t>" + IDENTIFIER + "</w:t>"

                                                    + "</w:r>"

		                                            + "<w:r>"
				                                            + "<w:t>after</w:t>"
				                                    + "</w:r>"
                                                    
                                            + "</w:p>"

                                    + "</w:tc>"

                                    + "<w:tc>"
                                            + "<w:tcPr>"
                                                    + "<w:tcW w:type=\"dxa\" w:w=\"3192\"/>"

                                            + "</w:tcPr>"

                                            + "<w:p>"
                                                    + "<w:pPr>"
                                                            + "<w:rPr>"
                                    + "<w:lang w:val=\"en-AU\"/>"

                                                            + "</w:rPr>"

                                                    + "</w:pPr>"

                                            + "</w:p>"

                                    + "</w:tc>"

                            + "</w:tr>"

                            + "<w:tr>"
                                    + "<w:tc>"
                                            + "<w:tcPr>"
                                                    + "<w:tcW w:type=\"dxa\" w:w=\"3192\"/>"

                                            + "</w:tcPr>"

                                            + "<w:p>"
                                                    + "<w:pPr>"
                                                            + "<w:rPr>"
                                    + "<w:lang w:val=\"en-AU\"/>"

                                                            + "</w:rPr>"

                                                    + "</w:pPr>"

                                            + "</w:p>"

                                    + "</w:tc>"

                                    + "<w:tc>"
                                            + "<w:tcPr>"
                                                    + "<w:tcW w:type=\"dxa\" w:w=\"3192\"/>"

                                            + "</w:tcPr>"

                                            + "<w:p>"
                                                    + "<w:pPr>"
                                                            + "<w:rPr>"
                                    + "<w:lang w:val=\"en-AU\"/>"

                                                            + "</w:rPr>"

                                                    + "</w:pPr>"

                                            + "</w:p>"

                                    + "</w:tc>"

                                    + "<w:tc>"
                                            + "<w:tcPr>"
                                                    + "<w:tcW w:type=\"dxa\" w:w=\"3192\"/>"

                                            + "</w:tcPr>"

                                            + "<w:p>"
                                                    + "<w:pPr>"
                                                            + "<w:rPr>"
                                    + "<w:lang w:val=\"en-AU\"/>"

                                                            + "</w:rPr>"

                                                    + "</w:pPr>"

                                            + "</w:p>"

                                    + "</w:tc>"

                            + "</w:tr>"

                    + "</w:tbl>"

                    + "<w:p>"
                            + "<w:pPr>"
                                    + "<w:rPr>"
                                            + "<w:lang w:val=\"en-AU\"/>"

                                    + "</w:rPr>"

                            + "</w:pPr>"

                    + "</w:p>"

                    + "<w:p>"
                            + "<w:pPr>"
                                    + "<w:rPr>"
                                            + "<w:lang w:val=\"en-AU\"/>"

                                    + "</w:rPr>"

                            + "</w:pPr>"

                            + "<w:r>"
                                    + "<w:rPr>"
                                            + "<w:lang w:val=\"en-AU\"/>"

                                    + "</w:rPr>"

                                    + "<w:t>Some more content</w:t>"

                            + "</w:r>"

                    + "</w:p>"

                    + "<w:sectPr>"
                            + "<w:pgSz w:h=\"15840\" w:w=\"12240\"/>"

                            + "<w:pgMar w:bottom=\"1440\" w:footer=\"708\" w:gutter=\"0\" w:header=\"708\" w:left=\"1440\" w:right=\"1440\" w:top=\"1440\"/>"

                            + "<w:cols w:space=\"708\"/>"

                            + "<w:docGrid w:linePitch=\"360\"/>"

                    + "</w:sectPr>"

            + "</w:body>"

    + "</w:document>";

//	private static int getIndex(List list, R run) {
//	
//	for (int i=0; i<list.size(); i++) {
//		
//		Object o = list.get(i);
//		if (o==run || (XmlUtils.unwrap(o)==run)) {
//			return i;
//		}
//		
//	}
//	return -1;
//}

}
